package dao.impl;

import dao.AccountDAO;
import dao.exception.PersistenceException;
import dao.impl.exception.UnableToConnectException;
import model.Account;

import java.sql.*;
import java.util.ArrayList;

public class AccountDaoMySQLImpl implements AccountDAO {

    private Connection connection;

    public AccountDaoMySQLImpl() throws UnableToConnectException {
        this.connection = new MySQLDbConnector().getDatabaseConnection();
    }

    /**
     * Creates customer-unique ID for accounts.
     * @param customerId ID of customer.
     * @return The whole account ID in "customerId-uniqueSequence" format.
     */
    private String getAccountIdForCustomer (int customerId) throws SQLException {
        PreparedStatement stmt = this.connection.prepareStatement(
                "SELECT COUNT(*)+1 AS new_account_id FROM customer INNER JOIN account " +
                        "ON customer.id=account.customerId WHERE customer.id=?");
        stmt.setInt(1, customerId);

        ResultSet rs = stmt.executeQuery();
        rs.next();

        return String.valueOf(customerId) + "-" + String.valueOf(1000 + rs.getInt("new_account_id"));
    }

    public void createAccount(Account account) throws PersistenceException {

        try {
            PreparedStatement stmt = this.connection.prepareStatement(
                    "INSERT INTO account (id, balance, isActive, customerId) VALUES (?, ?, ?, ?)");

            String accountId = this.getAccountIdForCustomer(account.getCustomerId());

            stmt.setString(1, accountId);
            stmt.setInt(2, account.getBalance());
            stmt.setBoolean(3, account.isActive());
            stmt.setInt(4, account.getCustomerId());

            stmt.execute();
        } catch (SQLException e) {
            throw new PersistenceException(e);
        }
    }

    public ArrayList<Account> readAccounts() throws PersistenceException {
        try {
            Statement stmt = this.connection.createStatement();
            ResultSet rs = stmt.executeQuery("SELECT * FROM account");
            ArrayList<Account> accounts = new ArrayList<Account>();

            while (rs.next()) {
                accounts.add(new Account(
                        rs.getString("id"),
                        rs.getInt("balance"),
                        rs.getInt("customerId"),
                        rs.getBoolean("isActive")
                ));
            }

            return accounts;
        } catch (SQLException e) {
            throw new PersistenceException(e);
        }
    }

    public Account readAccountById (String id) throws PersistenceException {
        try {
            PreparedStatement stmt = this.connection.prepareStatement("SELECT * FROM account WHERE id=? LIMIT 1");
            stmt.setString(1, id);

            ResultSet rs = stmt.executeQuery();
            rs.next();

            return new Account(
                rs.getString("id"),
                rs.getInt("balance"),
                rs.getInt("customerId"),
                rs.getBoolean("isActive")
            );
        } catch (SQLException e) {
            throw new PersistenceException(e);
        }
    }

    public void updateAccount(Account account) throws PersistenceException {
        try {
            if (account.getId() == null) {
                throw new PersistenceException("Unable to update account: id is missing.");
            }

            PreparedStatement stmt = this.connection.prepareStatement(
                    "UPDATE account SET balance=?, customerId=?, isActive=? WHERE id=?");
            stmt.setInt(1, account.getBalance());
            stmt.setInt(2, account.getCustomerId());
            stmt.setBoolean(3, account.isActive());
            stmt.setString(4, account.getId());

            stmt.execute();
        } catch (SQLException e) {
            throw new PersistenceException(e);
        }
    }

    private void addToBalance (String accountId, int amount) throws PersistenceException {
        try {
            PreparedStatement stmt = this.connection.prepareStatement("UPDATE account SET balance=balance+? WHERE id=?");
            stmt.setInt(1, amount);
            stmt.setString(2, accountId);
            stmt.execute();

            PreparedStatement logTransaction = this.connection.prepareStatement("INSERT INTO transaction (sourceAccount, amount, comment) VALUES (?, ?, ?)");
            logTransaction.setString(1, accountId);
            logTransaction.setInt(2, amount);
            logTransaction.setString(3, amount > 0 ? "Cash DEPOSIT" : "Cash WITHDRAW");
            logTransaction.execute();

        } catch (SQLException e) {
            e.printStackTrace();
        }
    }

    public void withdrawAccount(String accountId, int amount) throws PersistenceException {
        this.addToBalance(accountId, -amount);
    }

    public void depositAccount(String accountId, int amount) throws PersistenceException {
        this.addToBalance(accountId, amount);
    }

    public void deleteAccount(Account account) throws PersistenceException {
        try {
            PreparedStatement stmt = this.connection.prepareStatement("UPDATE account SET isActive=FALSE WHERE id=?");
            stmt.setString(1, account.getId());
            stmt.execute();

            PreparedStatement addMoneyWithdraw = this.connection.prepareStatement("INSERT INTO transaction (sourceAccount, amount, comment) VALUES (?, (SELECT balance FROM account WHERE id=?), ?)");
            addMoneyWithdraw.setString(1, account.getId());
            addMoneyWithdraw.setString(2, account.getId());
            addMoneyWithdraw.setString(3, "WITHDRAW on ACCOUNT CLOSE");
            addMoneyWithdraw.execute();

            PreparedStatement setZeroBalance = this.connection.prepareStatement("UPDATE account SET balance=0 WHERE id=?");
            setZeroBalance.setString(1, account.getId());
            setZeroBalance.execute();

        } catch (SQLException e) {
            throw new PersistenceException(e);
        }
    }
}
