package org.me.plugin.sample4.views;


import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Date;
import java.util.Properties;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.ViewPart;
import org.me.plugin.sample4.Activator;
import org.me.plugin.sample4.dialogs.EditDialog;


/**
 * This sample class demonstrates how to plug-in a new
 * workbench view. The view shows data obtained from the
 * model. The sample creates a dummy model on the fly,
 * but a real implementation would connect to the model
 * available either in this or another plug-in (e.g. the workspace).
 * The view is connected to the model using a content provider.
 * <p>
 * The view uses a label provider to define how model
 * objects should be presented in the view. Each
 * view can present the same model objects using
 * different labels and icons, if needed. Alternatively,
 * a single label provider can be shared between views
 * in order to ensure that objects of the same type are
 * presented in the same way everywhere.
 * <p>
 */

public class SampleView extends ViewPart {
	private static final String FILE_NAME = "c:\\test.properties";
	private Properties properties;
	private TableViewer viewer;
	private Action addAction;
	private Action editAction;
	private Action removeAction;
	private Action doubleClickAction;

	public TableViewer getTableViewer()
	{
		return viewer;
	}

	public Properties getProperties()
	{
		return properties;
	}
	
	/*
	 * The content provider class is responsible for
	 * providing objects to the view. It can wrap
	 * existing objects in adapters or simply return
	 * objects as-is. These objects may be sensitive
	 * to the current input of the view, or ignore
	 * it and always show the same content 
	 * (like Task List, for example).
	 */
	 
	class ViewContentProvider implements IStructuredContentProvider {
		public void inputChanged(Viewer v, Object oldInput, Object newInput) {
		}
		public void dispose() {
		}
		public Object[] getElements(Object parent) {
			return properties.keySet( ).toArray( );
		}
	}
	class ViewLabelProvider extends LabelProvider implements ITableLabelProvider {
		public String getColumnText(Object obj, int index) {
			if ( index == 0 )
				return obj.toString( );
			else
				return properties.getProperty( ( String ) obj );
		}
		public Image getColumnImage(Object obj, int index) {
			return getImage(obj);
		}
		public Image getImage(Object obj) {
			return null;
		}
	}

	@Override
	public void init( IViewSite site ) throws PartInitException
	{
		super.init( site );
		properties = new Properties();
		try
		{
			FileInputStream inputStream = new FileInputStream( FILE_NAME );
			properties.load( inputStream );
		}
		catch ( FileNotFoundException e )
		{
		}
		catch ( IOException e )
		{
			e.printStackTrace();
		}
	}

	@Override
	public void dispose( )
	{
		super.dispose( );
		try
		{
			FileOutputStream outputStream = new FileOutputStream( FILE_NAME );
			properties.store( outputStream, null );
		}
		catch ( FileNotFoundException e )
		{
			e.printStackTrace();
		}
		catch ( IOException e )
		{
			e.printStackTrace();
		}
	}
	
	/**
	 * The constructor.
	 */
	public SampleView() {
	}

	/**
	 * This is a callback that will allow us
	 * to create the viewer and initialize it.
	 */
	public void createPartControl(Composite parent) {
		viewer = new TableViewer(parent, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
		TableColumn tc = new TableColumn( viewer.getTable( ), SWT.NONE, 0 );
		tc.setText( "Key" );
		tc.setWidth( 200 );
		tc = new TableColumn( viewer.getTable( ), SWT.NONE, 1 );
		tc.setText( "Value" );
		tc.setWidth( 300 );
		viewer.setContentProvider(new ViewContentProvider());
		viewer.setLabelProvider(new ViewLabelProvider());
		viewer.setInput(getViewSite());
		viewer.getTable( ).setHeaderVisible( true );
		viewer.getTable( ).setLinesVisible( true );
		viewer.addSelectionChangedListener( new ISelectionChangedListener( )
		{

			public void selectionChanged( SelectionChangedEvent event )
			{
				IStructuredSelection selection = ( IStructuredSelection ) event.getSelection( );
				if ( selection.isEmpty( ) )
				{
					editAction.setEnabled( false );
					removeAction.setEnabled( false );
				}
				else
				{
					editAction.setEnabled( true );
					removeAction.setEnabled( true );
				}
			}
		} );
		makeActions();
		hookDoubleClickAction();
		contributeToActionBars();
	}

	private void contributeToActionBars() {
		IActionBars bars = getViewSite().getActionBars();
		fillLocalToolBar(bars.getToolBarManager());
	}

	private void fillLocalToolBar(IToolBarManager manager) {
		manager.add(addAction);
		manager.add(editAction);
		manager.add(removeAction);
	}

	private void makeActions() {
		doubleClickAction = new Action() {
			public void run() {
				ISelection selection = viewer.getSelection();
				Object obj = ((IStructuredSelection)selection).getFirstElement();
				showMessage("Double-click detected on "+obj.toString());
			}
		};
		addAction = new Action( "Add" )
		{

			public void run( )
			{
				try
				{
					Date date = new Date();
					properties.setProperty( "key " + date , "value " + date );
					viewer.refresh( );
				}
				catch ( Throwable e )
				{
					e.printStackTrace( );
				}
			}
		};
		addAction.setToolTipText( "Add" );
		addAction.setImageDescriptor( Activator.getImageDescriptor( "icons/add.gif" ) );
		editAction = new Action( "Edit" )
		{

			public void run( )
			{
				try
				{
					EditDialog dialog = new EditDialog( viewer.getTable( ).getShell( ) );
					dialog.open( SampleView.this );
				}
				catch ( Throwable e )
				{
					e.printStackTrace( );
				}
			}
		};
		editAction.setToolTipText( "Edit" );
		editAction.setEnabled( false );
		editAction.setImageDescriptor( Activator.getImageDescriptor( "icons/edit.gif" ) );
		editAction.setDisabledImageDescriptor( Activator.getImageDescriptor( "icons/edit_i.gif" ) );
		removeAction = new Action( "Remove" )
		{

			public void run( )
			{
				try
				{
					IStructuredSelection selection = ( IStructuredSelection ) viewer.getSelection( );
					String key = ( String ) selection.getFirstElement( );
					properties.remove( key );
					viewer.refresh( );
				}
				catch ( Throwable e )
				{
					e.printStackTrace( );
				}
			}
		};
		removeAction.setToolTipText( "Remove" );
		removeAction.setEnabled( false );
		removeAction.setImageDescriptor( Activator.getImageDescriptor( "icons/remove.gif" ) );
		removeAction.setDisabledImageDescriptor( Activator.getImageDescriptor( "icons/remove_i.gif" ) );
		
	}

	private void hookDoubleClickAction() {
		viewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(DoubleClickEvent event) {
				doubleClickAction.run();
			}
		});
	}
	private void showMessage(String message) {
		MessageDialog.openInformation(
			viewer.getControl().getShell(),
			"Sample View 2",
			message);
	}

	/**
	 * Passing the focus request to the viewer's control.
	 */
	public void setFocus() {
		viewer.getControl().setFocus();
	}
}